/*
 * Jalview - A Sequence Alignment Editor and Viewer (2.11.5.0)
 * Copyright (C) 2025 The Jalview Authors
 * 
 * This file is part of Jalview.
 * 
 * Jalview is free software: you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 
 * as published by the Free Software Foundation, either version 3
 * of the License, or (at your option) any later version.
 *  
 * Jalview is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty 
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 * PURPOSE.  See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with Jalview.  If not, see <http://www.gnu.org/licenses/>.
 * The Jalview Authors are detailed in the 'AUTHORS' file.
 */
package jalview.analysis;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.BitSet;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Vector;

import jalview.datamodel.AlignmentAnnotation;
import jalview.datamodel.PDBEntry;
import jalview.datamodel.SequenceGroup;
import jalview.datamodel.SequenceI;
import jalview.renderer.AnnotationRenderer;
import jalview.util.Constants;

public class AlignmentAnnotationUtils
{

  /**
   * Helper method to populate lists of annotation types for the Show/Hide
   * Annotations menus. If sequenceGroup is not null, this is restricted to
   * annotations which are associated with sequences in the selection group.
   * <p/>
   * If an annotation row is currently visible, its type (label) is added (once
   * only per type), to the shownTypes list. If it is currently hidden, it is
   * added to the hiddenTypesList.
   * <p/>
   * For rows that belong to a line graph group, so are always rendered
   * together:
   * <ul>
   * <li>Treat all rows in the group as visible, if at least one of them is</li>
   * <li>Build a list of all the annotation types that belong to the group</li>
   * </ul>
   * 
   * @param shownTypes
   *          a map, keyed by calcId (annotation source), whose entries are the
   *          lists of annotation types found for the calcId; each annotation
   *          type in turn may be a list (in the case of grouped annotations)
   * @param hiddenTypes
   *          a map, similar to shownTypes, but for hidden annotation types
   * @param annotations
   *          the annotations on the alignment to scan
   * @param forSequences
   *          the sequences to restrict search to
   */
  public static void getShownHiddenTypes(
          Map<String, List<List<String>>> shownTypes,
          Map<String, List<List<String>>> hiddenTypes,
          List<AlignmentAnnotation> annotations,
          List<SequenceI> forSequences)
  {
    BitSet visibleGraphGroups = AlignmentAnnotationUtils
            .getVisibleLineGraphGroups(annotations);

    /*
     * Build a lookup, by calcId (annotation source), of all annotation types in
     * each graph group.
     */
    Map<String, Map<Integer, List<String>>> groupLabels = new HashMap<String, Map<Integer, List<String>>>();

    // trackers for which calcId!label combinations we have dealt with
    List<String> addedToShown = new ArrayList<String>();
    List<String> addedToHidden = new ArrayList<String>();

    for (AlignmentAnnotation aa : annotations)
    {
      /*
       * Ignore non-positional annotations, can't render these against an
       * alignment
       */
      if (aa.annotations == null)
      {
        continue;
      }
      if (forSequences != null && (aa.sequenceRef != null
              && forSequences.contains(aa.sequenceRef)))
      {
        String calcId = aa.getCalcId();

        /*
         * Build a 'composite label' for types in line graph groups.
         */
        final List<String> labelAsList = new ArrayList<String>();
        final String displayLabel = aa.label;
        labelAsList.add(displayLabel);
        if (aa.graph == AlignmentAnnotation.LINE_GRAPH
                && aa.graphGroup > -1)
        {
          if (!groupLabels.containsKey(calcId))
          {
            groupLabels.put(calcId, new HashMap<Integer, List<String>>());
          }
          Map<Integer, List<String>> groupLabelsForCalcId = groupLabels
                  .get(calcId);
          if (groupLabelsForCalcId.containsKey(aa.graphGroup))
          {
            if (!groupLabelsForCalcId.get(aa.graphGroup)
                    .contains(displayLabel))
            {
              groupLabelsForCalcId.get(aa.graphGroup).add(displayLabel);
            }
          }
          else
          {
            groupLabelsForCalcId.put(aa.graphGroup, labelAsList);
          }
        }
        else
        /*
         * 'Simple case' - not a grouped annotation type - list of one label
         * only
         */
        {
          String rememberAs = calcId + "!" + displayLabel;
          if (aa.isForDisplay() && !addedToShown.contains(rememberAs)) // exclude noData annotations
          {
            if (!shownTypes.containsKey(calcId))
            {
              shownTypes.put(calcId, new ArrayList<List<String>>());
            }
            shownTypes.get(calcId).add(labelAsList);
            addedToShown.add(rememberAs);
          }
          else
          {
            if (!aa.visible && !addedToHidden.contains(rememberAs))
            {
              if (!hiddenTypes.containsKey(calcId))
              {
                hiddenTypes.put(calcId, new ArrayList<List<String>>());
              }
              hiddenTypes.get(calcId).add(labelAsList);
              addedToHidden.add(rememberAs);
            }
          }
        }
      }
    }
    /*
     * Finally add the 'composite group labels' to the appropriate lists,
     * depending on whether the group is identified as visible or hidden. Don't
     * add the same label more than once (there may be many graph groups that
     * generate it).
     */
    for (String calcId : groupLabels.keySet())
    {
      for (int group : groupLabels.get(calcId).keySet())
      {
        final List<String> groupLabel = groupLabels.get(calcId).get(group);
        // don't want to duplicate 'same types in different order'
        Collections.sort(groupLabel);
        if (visibleGraphGroups.get(group))
        {
          if (!shownTypes.containsKey(calcId))
          {
            shownTypes.put(calcId, new ArrayList<List<String>>());
          }
          if (!shownTypes.get(calcId).contains(groupLabel))
          {
            shownTypes.get(calcId).add(groupLabel);
          }
        }
        else
        {
          if (!hiddenTypes.containsKey(calcId))
          {
            hiddenTypes.put(calcId, new ArrayList<List<String>>());
          }
          if (!hiddenTypes.get(calcId).contains(groupLabel))
          {
            hiddenTypes.get(calcId).add(groupLabel);
          }
        }
      }
    }
  }

  /**
   * Updates the lists of shown and hidden secondary structure types based on
   * the selected sequence group.
   * 
   * @param shownTypes
   *          A list that will be populated with the providers of secondary
   *          structures that are shown.
   * @param hiddenTypes
   *          A list that will be populated with the providers of secondary
   *          structures that are hidden.
   * @param annotations
   *          A list of AlignmentAnnotation objects.
   * @param selectedSequenceGroup
   *          The sequence group selected by the user.
   */
  public static void getShownHiddenSecondaryStructureProvidersForGroup(
          List<String> shownTypes, List<String> hiddenTypes,
          List<AlignmentAnnotation> annotations,
          SequenceGroup selectedSequenceGroup)
  {
    // Return if the selected sequence group or annotations are null
    if (selectedSequenceGroup == null || annotations == null)
    {
      return;
    }

    // Get the secondary structure sources of the selected sequence group
    List<String> ssSourcesForSelectedGroup = selectedSequenceGroup
            .getSecondaryStructureSources();

    // Return if there are no secondary structure sources for the selected group
    if (ssSourcesForSelectedGroup == null
            || ssSourcesForSelectedGroup.isEmpty())
    {
      return;
    }

    // Iterate through each annotation
    for (AlignmentAnnotation aa : annotations)
    {
      /* Skip to the next annotation if the annotation, the annotation's group 
       * reference is null, or the annotation's group reference does not match 
       * the selected group 
       */
      if (aa.annotations == null || aa.groupRef == null
              || selectedSequenceGroup != aa.groupRef
              || !aa.label.startsWith(
                      Constants.SECONDARY_STRUCTURE_CONSENSUS_LABEL))
      {
        continue;
      }

      /* Find a provider from the secondary structure sources that matches 
       * the annotation's label. This is to exclude secondary structure 
       * providers which has no secondary structure data for the selected group.
       */
      Optional<String> provider = ssSourcesForSelectedGroup.stream()
              .filter(aa.label::contains).findFirst()
              .map(substring -> aa.label.substring(0,
                      aa.label.indexOf(substring) + substring.length()));

      // If a matching provider is found and the annotation is visible, add
      // the provider to the shown types list (if not already in shownTypes).
      // If the annotation is not visible, add it to hiddenTypes list.
      provider.ifPresent(p -> {
        if (aa.visible && !shownTypes.contains(p))
        {
          shownTypes.add(p);
        }
        else if (!aa.visible && !shownTypes.contains(p))
        {
          hiddenTypes.add(p);
        }
      });
    }
  }

  /**
   * Returns a BitSet (possibly empty) of those graphGroups for line graph
   * annotations, which have at least one member annotation row marked visible.
   * <p/>
   * Only one row in each visible group is marked visible, but when it is drawn,
   * so are all the other rows in the same group.
   * <p/>
   * This lookup set allows us to check whether rows apparently marked not
   * visible are in fact shown.
   * 
   * @see AnnotationRenderer#drawComponent
   * @param annotations
   * @return
   */
  public static BitSet getVisibleLineGraphGroups(
          List<AlignmentAnnotation> annotations)
  {
    BitSet result = new BitSet();
    for (AlignmentAnnotation ann : annotations)
    {
      if (ann.graph == AlignmentAnnotation.LINE_GRAPH && ann.visible) 
      {
        int gg = ann.graphGroup;
        if (gg > -1)
        {
          result.set(gg);
        }
      }
    }
    return result;
  }

  /**
   * Converts an array of AlignmentAnnotation into a List of
   * AlignmentAnnotation. A null array is converted to an empty list.
   * 
   * @param anns
   * @return
   */
  public static List<AlignmentAnnotation> asList(AlignmentAnnotation[] anns)
  {
    // TODO use AlignmentAnnotationI instead when it exists
    return (anns == null ? Collections.<AlignmentAnnotation> emptyList()
            : Arrays.asList(anns));
  }

  /**
   * Pulls sequence associated annotation on an alignment sequence onto its
   * dataset sequence and leaves the alignment seuqence's annotation as 'added
   * reference annotation'
   * 
   * NB. This will overwrite/delete any existing annotation on the dataset
   * sequence with matching calcId and label
   * 
   * @param newAnnot
   * @param typeName
   * @param calcId
   * @param aSeq
   */
  public static void replaceAnnotationOnAlignmentWith(
          AlignmentAnnotation newAnnot, String typeName, String calcId,
          SequenceI aSeq)
  {
    SequenceI dsseq = aSeq.getDatasetSequence();
    while (dsseq.getDatasetSequence() != null)
    {
      dsseq = dsseq.getDatasetSequence();
    }
    // look for same annotation on dataset and lift this one over
    List<AlignmentAnnotation> dsan = dsseq.getAlignmentAnnotations(calcId,
            typeName);
    if (dsan != null && dsan.size() > 0)
    {
      for (AlignmentAnnotation dssan : dsan)
      {
        dsseq.removeAlignmentAnnotation(dssan);
      }
    }
    AlignmentAnnotation dssan = new AlignmentAnnotation(newAnnot);
    dsseq.addAlignmentAnnotation(dssan);
    dssan.adjustForAlignment();
  }
  
  /**
   * Looks for (and materialises) metadata for the given secondary structure annotation
   *  - sets a property Constants.ANNOTATION_DETAILS 
   *  -- giving structure ID:chain for 3D structure data
   *  --   
   * @param aa
   * @return Secondary Structure 'provider' 
   */
  public static String extractSSSourceFromAnnotationDescription(
          AlignmentAnnotation aa)
  {
    if (aa == null || aa.label == null)
    {
      return null;
    }

    String label = aa.label;

    // Proceed if the label exists in SECONDARY_STRUCTURE_LABELS
    if (!Constants.SECONDARY_STRUCTURE_LABELS.containsKey(label))
    {
      return null;
    }

    // Check if the annotation has a provider saved as a property
    String provider = aa.getProperty(Constants.SS_PROVIDER_PROPERTY);
    if (provider != null)
    {
      
      if( Constants.PDB.equals(aa.getProperty(Constants.SS_PROVIDER_PROPERTY)) 
              && aa.getProperty(Constants.PDBID) != null 
              && !aa.hasAnnotationDetailsProperty() )
      {
        String annotDetails = aa.getProperty(Constants.PDBID);
        if(aa.getProperty(Constants.CHAINID) != null)
        {
          annotDetails = annotDetails + ":" + aa.getProperty(Constants.CHAINID);
        }
        
        aa.setAnnotationDetailsProperty(annotDetails); 
        
      }
      
      return provider;
    }

    // JPred label
    if (Constants.SS_ANNOTATION_FROM_JPRED_LABEL.equals(label))
    {
      provider = Constants.SECONDARY_STRUCTURE_LABELS.get(label);
      aa.setProperty(Constants.SS_PROVIDER_PROPERTY, provider);
      return provider;
    }

    // 3D structure label
    if (aa.description != null
            && Constants.SS_ANNOTATION_LABEL.equals(label)
            && Constants.SS_ANNOTATION_LABEL.equals(aa.description))
    {
      provider = Constants.SECONDARY_STRUCTURE_LABELS.get(label);
      aa.setProperty(Constants.SS_PROVIDER_PROPERTY, provider);
      return provider;
    }

    // Identify provider from sequence reference, description and PDB entries
    if (aa.sequenceRef == null
            || aa.sequenceRef.getDatasetSequence() == null)
    {
      return null;
    }

    Vector<PDBEntry> pdbEntries = aa.sequenceRef.getDatasetSequence()
            .getAllPDBEntries();
    if (pdbEntries == null || pdbEntries.isEmpty())
    {
      return null;
    }

    for (PDBEntry entry : pdbEntries)
    {
      if (entry == null || entry.getId() == null)
      {
        continue;
      }

      String entryProvider = entry.getProvider();

      if (entryProvider == null)
      {
        // No provider - so this is either an old Jalview project, or not
        // retrieved from recognised source
        entryProvider = Constants.PDB;
      }

      // Should (re)use a standard mechanism for extracting the PDB ID as it
      // is written 1QWXTUV:CHAIN
      String entryId = entry.getId();
      int colonIndex = entryId.indexOf(':');
      // Check if colon exists
      if (colonIndex != -1)
      {
        // Trim the string from first occurrence of colon
        entryId = entryId.substring(0, colonIndex);
      }

      // Annotation from PDB (description text match)
      if (Constants.PDB.equals(entryProvider) && aa.description != null
              && aa.description.toLowerCase().contains(
                      "secondary structure for " + entryId.toLowerCase()))
      {

        aa.setProperty(Constants.SS_PROVIDER_PROPERTY, Constants.PDB);

        String annotDetails = aa.getProperty(Constants.PDBID);
        if (annotDetails != null
                && aa.getProperty(Constants.CHAINID) != null)
        {
          annotDetails += ":" + aa.getProperty(Constants.CHAINID);
        }

        if (annotDetails != null)
        {
          aa.setAnnotationDetailsProperty(annotDetails);
        }

        return Constants.PDB;
      }

      // Annotation from other providers (AlphaFold, SwissModel)
      if (!Constants.PDB.equals(entryProvider) 
              && aa.description.toLowerCase().contains(entryId.toLowerCase()))
      {
        aa.setProperty(Constants.SS_PROVIDER_PROPERTY, entryProvider);
        return entryProvider;
      }

    }

    return null;

  }
}
